<?php

namespace App\Repositories\Backend\Auth;

use App\Repositories\BaseRepository;
use App\Models\Auth\Permission;
use Illuminate\Support\Facades\DB;

/**
 * Class PermissionRepository.
 */
class PermissionRepository extends BaseRepository
{
    /**
     * PermissionRepository constructor.
     *
     * @param  Permission  $model
     */
    public function __construct(Permission $model)
    {
        $this->model = $model;
    }

    /**
     * @param array $data
     *
     * @throws GeneralException
     * @throws \Throwable
     * @return Permission
     */
    public function create(array $data) : Permission
    {
        // Make sure it doesn't already exist
        if ($this->permissionExists($data['name'])) {
            throw new GeneralException('A permission already exists with the name '.e($data['name']));
        }

        

        return DB::transaction(function () use ($data) {
            $permission = $this->model::create(['name' => strtolower($data['name']),
            'guard_name' => 'web'
            ]);

            if ($permission) {
              

                return $permission;
            }

            throw new GeneralException(trans('exceptions.backend.access.permissions.create_error'));
        });
    }

    /**
     * @param Role  $role
     * @param array $data
     *
     * @throws GeneralException
     * @throws \Throwable
     * @return mixed
     */
    public function update(Permission $permission, array $data)
    {
        

        // If the name is changing make sure it doesn't already exist
        if ($permission->name !== strtolower($data['name'])) {
            if ($this->permissionExists($data['name'])) {
                throw new GeneralException('A permission already exists with the name '.$data['name']);
            }
        }


        return DB::transaction(function () use ($permission, $data) {
            if ($permission->update([
                'name' => strtolower($data['name']),
            ])) {
               
                return $permission;
            }

            throw new GeneralException(trans('exceptions.backend.access.permissions.update_error'));
        });
    }

    /**
     * @param $name
     *
     * @return bool
     */
    protected function permissionExists($name) : bool
    {
        return $this->model
            ->where('name', strtolower($name))
            ->count() > 0;
    }
}
